// Background script for Telegram Bot Token Scanner

// Storage keys
const STORAGE_KEYS = {
  TOKENS: 'telegram_tokens',
  SETTINGS: 'scanner_settings'
};

// Default settings
const DEFAULT_SETTINGS = {
  autoValidate: true,
  maxHistory: 100,
  showNotifications: true,
  // Background scanning settings
  enabled: true,
  interval: 5000, // 5 seconds
  scanOnNavigation: true,
  scanNetworkRequests: true,
  scanPageContent: true,
  scanURL: true,
  // Filescan.io API settings
  filescanEnabled: true,
  filescanApiKey: 'WCUMv9FjuWfqkE3GdCGm1Y7rkrwIzxOAN9tzNEAS',
  filescanInterval: 300000, // 5 minutes
  filescanLastCheck: 0
};

// Initialize storage
async function initializeStorage() {
  const result = await chrome.storage.local.get([STORAGE_KEYS.TOKENS, STORAGE_KEYS.SETTINGS]);
  
  if (!result[STORAGE_KEYS.TOKENS]) {
    await chrome.storage.local.set({ [STORAGE_KEYS.TOKENS]: [] });
  }
  
  if (!result[STORAGE_KEYS.SETTINGS]) {
    console.log('Initializing default settings:', DEFAULT_SETTINGS);
    await chrome.storage.local.set({ [STORAGE_KEYS.SETTINGS]: DEFAULT_SETTINGS });
  } else {
    console.log('Existing settings found:', result[STORAGE_KEYS.SETTINGS]);
  }
}

// Search Filescan.io for Telegram bot tokens
async function searchFilescanForTokens() {
  const settings = await getSettings();
  
  console.log('Filescan.io settings:', {
    enabled: settings.filescanEnabled,
    hasApiKey: !!settings.filescanApiKey,
    lastCheck: settings.filescanLastCheck,
    interval: settings.filescanInterval
  });
  
  if (!settings.filescanEnabled || !settings.filescanApiKey) {
    console.log('Filescan.io search disabled or API key not configured');
    return;
  }
  
  const now = Date.now();
  if (now - settings.filescanLastCheck < settings.filescanInterval) {
    console.log('Filescan.io search too recent, skipping');
    return;
  }
  
  try {
    console.log('Searching Filescan.io for Telegram bot tokens...');
    
    // Search queries for different Telegram bot token patterns
    const searchQueries = [
      'api.telegram.org/bot',
      'telegram bot token',
      'bot token telegram',
      'telegram api key',
      'telegram bot api'
    ];
    
    const foundTokens = [];
    
    for (const query of searchQueries) {
      try {
        // Use the correct Filescan.io API endpoint for searching reports
        const response = await fetch(`https://www.filescan.io/api/reports/search?query=${encodeURIComponent(query)}&limit=50`, {
          method: 'GET',
          headers: {
            'X-Api-Key': settings.filescanApiKey,
            'Content-Type': 'application/json'
          }
        });
        
        if (response.ok) {
          const data = await response.json();
          
          if (data && data.results) {
            for (const result of data.results) {
              // Extract tokens from basic report data
              const tokens = extractTokensFromFilescanResult(result);
              foundTokens.push(...tokens);
              
              // Try to get more detailed content from the report
              try {
                const detailedTokens = await getDetailedReportContent(result.sha256, settings.filescanApiKey);
                foundTokens.push(...detailedTokens);
              } catch (detailError) {
                console.debug('Could not fetch detailed report content:', detailError);
              }
            }
          }
        } else {
          console.error('Filescan.io API error:', response.status, response.statusText);
        }
      } catch (error) {
        console.error('Error searching Filescan.io:', error);
      }
      
      // Add delay between requests to avoid rate limiting
      await new Promise(resolve => setTimeout(resolve, 1000));
    }
    
    // Process found tokens
    if (foundTokens.length > 0) {
      console.log(`Found ${foundTokens.length} potential tokens from Filescan.io`);
      
      for (const tokenData of foundTokens) {
        await saveToken({
          ...tokenData,
          source: 'filescan.io',
          timestamp: Date.now()
        });
      }
    }
    
    // Update last check time
    const updatedSettings = { ...settings, filescanLastCheck: now };
    await chrome.storage.local.set({ [STORAGE_KEYS.SETTINGS]: updatedSettings });
    
  } catch (error) {
    console.error('Error in Filescan.io search:', error);
  }
}

// Extract tokens from Filescan.io search result
function extractTokensFromFilescanResult(result) {
  const tokens = [];
  
  // Extract from report summary/tags
  if (result.summary) {
    const summaryTokens = extractTokensFromText(result.summary);
    for (const token of summaryTokens) {
      tokens.push({
        token: token,
        url: `https://www.filescan.io/reports/${result.sha256}`,
        title: `Filescan.io: ${result.filename || 'Unknown file'}`,
        timestamp: Date.now(),
        source: 'filescan.io',
        fileInfo: {
          sha256: result.sha256,
          fileName: result.filename,
          fileSize: result.file_size,
          fileType: result.file_type,
          verdict: result.verdict,
          tags: result.tags
        }
      });
    }
  }
  
  // Extract from filename
  if (result.filename) {
    const nameTokens = extractTokensFromText(result.filename);
    for (const token of nameTokens) {
      tokens.push({
        token: token,
        url: `https://www.filescan.io/reports/${result.sha256}`,
        title: `Filescan.io: ${result.filename}`,
        timestamp: Date.now(),
        source: 'filescan.io',
        fileInfo: {
          sha256: result.sha256,
          fileName: result.filename,
          fileSize: result.file_size,
          fileType: result.file_type,
          verdict: result.verdict,
          tags: result.tags
        }
      });
    }
  }
  
  // Extract from tags if available
  if (result.tags && Array.isArray(result.tags)) {
    const tagText = result.tags.join(' ');
    const tagTokens = extractTokensFromText(tagText);
    for (const token of tagTokens) {
      tokens.push({
        token: token,
        url: `https://www.filescan.io/reports/${result.sha256}`,
        title: `Filescan.io: ${result.filename || 'Unknown file'} (tags)`,
        timestamp: Date.now(),
        source: 'filescan.io',
        fileInfo: {
          sha256: result.sha256,
          fileName: result.filename,
          fileSize: result.file_size,
          fileType: result.file_type,
          verdict: result.verdict,
          tags: result.tags
        }
      });
    }
  }
  
  return tokens;
}

// Extract tokens from text (reused from content script logic)
function extractTokensFromText(text) {
  const tokens = new Set();
  
  // Telegram bot token patterns
  const patterns = [
    /\b(\d{8,10}:[A-Za-z0-9_-]{35})\b/g,
    /\/bot(\d{8,10}:[A-Za-z0-9_-]{35})/g,
    /(\d{8,10}:[A-Za-z0-9_-]{35})/g
  ];
  
  patterns.forEach(pattern => {
    let match;
    while ((match = pattern.exec(text)) !== null) {
      let token = match[1] || match[0];
      token = token.replace(/['""]/g, '').trim();
      
      if (/^\d{8,10}:[A-Za-z0-9_-]{35}$/.test(token)) {
        tokens.add(token);
      }
    }
  });
  
  return Array.from(tokens);
}

// Get detailed report content from Filescan.io
async function getDetailedReportContent(reportId, apiKey) {
  const tokens = [];
  
  try {
    // Get all files in the report
    const filesResponse = await fetch(`https://www.filescan.io/api/reports/${reportId}/files`, {
      method: 'GET',
      headers: {
        'X-Api-Key': apiKey,
        'Content-Type': 'application/json'
      }
    });
    
    if (filesResponse.ok) {
      const filesData = await filesResponse.json();
      
      if (filesData && Array.isArray(filesData)) {
        for (const file of filesData) {
          // Get specific file content if it's a text file
          if (file.file_type && (
            file.file_type.includes('text') || 
            file.file_type.includes('json') ||
            file.file_type.includes('xml') ||
            file.file_type.includes('log')
          )) {
            try {
              const fileResponse = await fetch(`https://www.filescan.io/api/reports/${reportId}/${file.sha256}`, {
                method: 'GET',
                headers: {
                  'X-Api-Key': apiKey,
                  'Content-Type': 'application/json'
                }
              });
              
              if (fileResponse.ok) {
                const fileData = await fileResponse.json();
                
                // Extract tokens from file content
                if (fileData.content) {
                  const contentTokens = extractTokensFromText(fileData.content);
                  for (const token of contentTokens) {
                    tokens.push({
                      token: token,
                      url: `https://www.filescan.io/reports/${reportId}/${file.sha256}`,
                      title: `Filescan.io: ${file.filename || 'Unknown file'}`,
                      timestamp: Date.now(),
                      source: 'filescan.io',
                      fileInfo: {
                        sha256: file.sha256,
                        fileName: file.filename,
                        fileSize: file.file_size,
                        fileType: file.file_type,
                        reportId: reportId
                      }
                    });
                  }
                }
              }
            } catch (fileError) {
              console.debug('Could not fetch file content:', fileError);
            }
          }
        }
      }
    }
  } catch (error) {
    console.debug('Error fetching detailed report content:', error);
  }
  
  return tokens;
}

// Validate a Telegram bot token
async function validateToken(token) {
  try {
    const response = await fetch(`https://api.telegram.org/bot${token}/getMe`);
    const data = await response.json();
    
    if (data.ok) {
      return {
        valid: true,
        botInfo: data.result,
        error: null
      };
    } else {
      return {
        valid: false,
        botInfo: null,
        error: data.description || 'Unknown error'
      };
    }
  } catch (error) {
    return {
      valid: false,
      botInfo: null,
      error: error.message
    };
  }
}

// Save token to storage
async function saveToken(tokenData) {
  const result = await chrome.storage.local.get(STORAGE_KEYS.TOKENS);
  const tokens = result[STORAGE_KEYS.TOKENS] || [];
  
  // Check if token already exists
  const existingTokenIndex = tokens.findIndex(t => t.token === tokenData.token);
  
  if (existingTokenIndex !== -1) {
    // Update existing token
    tokens[existingTokenIndex] = {
      ...tokens[existingTokenIndex],
      ...tokenData,
      lastSeen: Date.now(),
      occurrences: (tokens[existingTokenIndex].occurrences || 1) + 1
    };
  } else {
    // Add new token
    tokens.push({
      ...tokenData,
      firstSeen: Date.now(),
      lastSeen: Date.now(),
      occurrences: 1
    });
  }
  
  // Sort by last seen (most recent first)
  tokens.sort((a, b) => b.lastSeen - a.lastSeen);
  
  // Limit history size
  const settings = await getSettings();
  if (tokens.length > settings.maxHistory) {
    tokens.splice(settings.maxHistory);
  }
  
  await chrome.storage.local.set({ [STORAGE_KEYS.TOKENS]: tokens });
  
  return tokens;
}

// Get settings
async function getSettings() {
  const result = await chrome.storage.local.get(STORAGE_KEYS.SETTINGS);
  return { ...DEFAULT_SETTINGS, ...result[STORAGE_KEYS.SETTINGS] };
}

// Show notification
async function showNotification(token, isValid, botInfo) {
  try {
    const settings = await getSettings();
    if (!settings.showNotifications) return;
    
    const title = isValid ? 'Valid Telegram Bot Found' : 'Invalid Telegram Bot Token';
    const message = isValid 
      ? `Bot: ${botInfo.first_name} (@${botInfo.username})`
      : `Token: ${token.substring(0, 20)}...`;
    
    // Check if notifications API is available
    if (chrome.notifications && chrome.notifications.create) {
      chrome.notifications.create({
        type: 'basic',
        title: title,
        message: message
      });
    } else {
      console.log('Notification:', title, '-', message);
    }
  } catch (error) {
    console.error('Error showing notification:', error);
  }
}

// Process found tokens
async function processTokens(message) {
  const { tokens, url, title, timestamp } = message;
  const settings = await getSettings();
  
  for (const token of tokens) {
    let validationResult = null;
    
    if (settings.autoValidate) {
      validationResult = await validateToken(token);
    }
    
    const tokenData = {
      token: token,
      url: url,
      pageTitle: title,
      timestamp: timestamp,
      validation: validationResult
    };
    
    await saveToken(tokenData);
    
    // Show notification if token is valid
    if (validationResult && validationResult.valid) {
      await showNotification(token, true, validationResult.botInfo);
    }
  }
}

// Notify all content scripts about settings update
async function notifyContentScripts(settings) {
  try {
    const tabs = await chrome.tabs.query({});
    for (const tab of tabs) {
      try {
        await chrome.tabs.sendMessage(tab.id, {
          type: 'SETTINGS_UPDATED',
          settings: settings
        });
      } catch (error) {
        // Ignore errors for tabs without content script
      }
    }
  } catch (error) {
    console.error('Error notifying content scripts:', error);
  }
}

// Message listener
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.type === 'TOKENS_FOUND') {
    processTokens(message);
    sendResponse({ success: true });
  } else if (message.type === 'GET_TOKENS') {
    chrome.storage.local.get(STORAGE_KEYS.TOKENS).then(result => {
      sendResponse({ tokens: result[STORAGE_KEYS.TOKENS] || [] });
    });
    return true; // Keep message channel open for async response
  } else if (message.type === 'GET_SETTINGS') {
    getSettings().then(settings => {
      sendResponse({ settings });
    });
    return true; // Keep message channel open for async response
  } else if (message.type === 'UPDATE_SETTINGS') {
    chrome.storage.local.set({ [STORAGE_KEYS.SETTINGS]: message.settings }).then(async () => {
      // Notify all content scripts about the settings update
      await notifyContentScripts(message.settings);
      sendResponse({ success: true });
    });
    return true; // Keep message channel open for async response
  } else if (message.type === 'VALIDATE_TOKEN') {
    validateToken(message.token).then(result => {
      sendResponse(result);
    });
    return true; // Keep message channel open for async response
  } else if (message.type === 'CLEAR_HISTORY') {
    chrome.storage.local.set({ [STORAGE_KEYS.TOKENS]: [] }).then(() => {
      sendResponse({ success: true });
    });
    return true; // Keep message channel open for async response
  } else if (message.type === 'SEARCH_FILESCAN') {
    searchFilescanForTokens().then(() => {
      sendResponse({ success: true });
    });
    return true; // Keep message channel open for async response
  }
});

// Periodic Filescan.io search
let filescanInterval = null;

function startFilescanSearch() {
  if (filescanInterval) {
    clearInterval(filescanInterval);
  }
  
  // Search immediately on startup
  searchFilescanForTokens();
  
  // Set up periodic search every 5 minutes
  filescanInterval = setInterval(() => {
    searchFilescanForTokens();
  }, 300000); // 5 minutes
}

function stopFilescanSearch() {
  if (filescanInterval) {
    clearInterval(filescanInterval);
    filescanInterval = null;
  }
}

// Initialize on startup
chrome.runtime.onStartup.addListener(async () => {
  await initializeStorage();
  startFilescanSearch();
});

chrome.runtime.onInstalled.addListener(async () => {
  await initializeStorage();
  startFilescanSearch();
});

// Initialize immediately
initializeStorage().then(() => {
  startFilescanSearch();
});
